<?php
/**
 * OnGetSellingUnitModelEventListener.php 2020-3-18
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */

declare(strict_types=1);

namespace Gambio\Shop\Attributes\SellingUnitModel\Database\Listener;

use Gambio\Shop\Attributes\SellingUnitModel\Database\Exceptions\OptionValueIdDoesNotExistsException;
use Gambio\Shop\Attributes\SellingUnitModel\Database\Repository\DTO\AttributesModelDto;
use Gambio\Shop\Attributes\SellingUnitModel\Database\Repository\DTO\OptionValuesIdDto;
use Gambio\Shop\Attributes\SellingUnitModel\Database\Service\ReadServiceInterface;
use Gambio\Shop\ProductModifiers\Modifiers\ValueObjects\ModifierIdentifierInterface;
use Gambio\Shop\SellingUnit\Database\Unit\Events\OnGetSellingUnitModelEvent;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Model;

/**
 * Class OnGetSellingUnitModelEventListener
 * @package Gambio\Shop\Attributes\SellingUnitModel\Database\Listener
 */
class OnGetSellingUnitModelEventListener
{
    /**
     * @var ReadServiceInterface
     */
    protected $service;
    
    
    /**
     * OnGetSellingUnitModelEventListener constructor.
     *
     * @param ReadServiceInterface $service
     */
    public function __construct(ReadServiceInterface $service)
    {
        $this->service = $service;
    }
    
    
    /**
     * @param OnGetSellingUnitModelEvent $event
     */
    public function __invoke(OnGetSellingUnitModelEvent $event)
    {
        $modifiers = $event->modifierIdentifiers();
        
        if (count($modifiers)) {
            
            $model = $event->model()->value();
            foreach ($modifiers as $modifier) {
                
                if ($modifier->type() === 'attribute') {
                    
                    $modelDto = $this->modelDto($modifier);
                    
                    if ($modelDto->model() === '') {
                        
                        continue;
                    }
                    
                    $model .= '-' . $modelDto->model();
                }
            }
            
            $event->setModel(new Model($model));
        }
    }
    
    
    /**
     * @param ModifierIdentifierInterface $modifier
     *
     * @return AttributesModelDto
     */
    protected function modelDto(ModifierIdentifierInterface $modifier):  AttributesModelDto
    {
        $dto = new OptionValuesIdDto((int)$modifier->value());
        try {
            $result = $this->service->attributeModelByOptionValuesId($dto);
        } catch (OptionValueIdDoesNotExistsException $exception) {
            unset($exception);
            // if the model is empty it gets skipped
            $result = new AttributesModelDto('');
        }
        
        return $result;
    }
}